#include "StdV3D.h"

#include "Basics/Vector.h"
#include "Basics/MatrixTransf.h"
#include "Basics/Maths.h"

#include "IsoSurface.h"
#include "TriangulatorMarchCubes.h"

namespace V3D {

TriangulatorMarchCubes::TriangulatorMarchCubes(const Vector3D& vcCenterGrid, const Vector3D& vcSize, 
											   int32 nDivideX, int32 nDivideY, int32 nDivideZ)
{
	SetGridInfo(vcCenterGrid, vcSize, nDivideX, nDivideY, nDivideZ);
}

TriangulatorMarchCubes::~TriangulatorMarchCubes()
{}


void TriangulatorMarchCubes::SetGridInfo( const Vector3D& vcCenterGrid, const Vector3D& vcSize, 
										  int32 nDivideX, int32 nDivideY, int32 nDivideZ)
{
	m_nDivideX = nDivideX;
	m_nDivideY = nDivideY;
	m_nDivideZ = nDivideZ;
	m_vcMarchCubeSize.Set( float( vcSize.x / m_nDivideX ),
						   float( vcSize.y / m_nDivideY ),
						   float( vcSize.z / m_nDivideZ ) );
	m_vcCornerPos = vcCenterGrid - (vcSize * .5f);
}


Vector3D TriangulatorMarchCubes::GetGridPoint(int32 nPointX, int32 nPointY, int32 nPointZ) const
{
	Vector3D vcResult = m_vcCornerPos;
	vcResult.x += float(nPointX) * m_vcMarchCubeSize.x;
	vcResult.y += float(nPointY) * m_vcMarchCubeSize.y;
	vcResult.z += float(nPointZ) * m_vcMarchCubeSize.z;
	return vcResult;
}

Vector3D TriangulatorMarchCubes::GetSize() const
{
	Vector3D vcSize;
	vcSize.x = m_vcMarchCubeSize.x*float(m_nDivideX);
	vcSize.y = m_vcMarchCubeSize.y*float(m_nDivideY);
	vcSize.z = m_vcMarchCubeSize.z*float(m_nDivideZ);
	return vcSize;

}


void TriangulatorMarchCubes::ComputeUsefulSubGrid( int32& nStartCubeX, int32& nCubesX,
												   int32& nStartCubeY, int32& nCubesY,
												   int32& nStartCubeZ, int32& nCubesZ,
												   const IsoSurface& iso) const
{
	Vector3D vcMinIso, vcMaxIso;
	bool bIsoSurfaceBounded = iso.ComputeBounds(vcMinIso, vcMaxIso);
	int32 nMaxCubesX = GetDivideX();
	int32 nMaxCubesY = GetDivideY();
	int32 nMaxCubesZ = GetDivideZ();

	if( ! bIsoSurfaceBounded)
	{
		nStartCubeX = 0;
		nStartCubeY = 0;
		nStartCubeZ = 0;
		nCubesX = nMaxCubesX;
		nCubesY = nMaxCubesY;
		nCubesZ = nMaxCubesZ;
	}
	else
	{
		vcMinIso -= m_vcCornerPos;
		nStartCubeX = Max( 0, Floor( float(vcMinIso.x / m_vcMarchCubeSize.x)) );
		nStartCubeY = Max( 0, Floor( float(vcMinIso.y / m_vcMarchCubeSize.y)) );
		nStartCubeZ = Max( 0, Floor( float(vcMinIso.z / m_vcMarchCubeSize.z)) );

		vcMaxIso -= m_vcCornerPos;
		nCubesX = Min( nMaxCubesX, Ceil( float(vcMaxIso.x / m_vcMarchCubeSize.x)) ) - nStartCubeX;
		nCubesY = Min( nMaxCubesY, Ceil( float(vcMaxIso.y / m_vcMarchCubeSize.y)) ) - nStartCubeY;
		nCubesZ = Min( nMaxCubesZ, Ceil( float(vcMaxIso.z / m_vcMarchCubeSize.z)) ) - nStartCubeZ;
	}
}




// Table des configurations de cubes
const TriangulatorMarchCubes::SCubeConf TriangulatorMarchCubes::k_aCubeConfs[256] = 
{
	{0, {{-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{1, {{ 8,  0,  3}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{1, {{ 1,  0,  9}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{2, {{ 8,  1,  3}, { 8,  9,  1}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{1, {{ 2,  1, 10}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{2, {{ 8,  0,  3}, { 2,  1, 10}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{2, {{ 2,  9, 10}, { 2,  0,  9}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 8,  2,  3}, {10,  2,  8}, { 9, 10,  8}, {-1, -1, -1}, {-1, -1, -1}}},
	{1, {{11,  3,  2}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{2, {{11,  0,  2}, {11,  8,  0}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{2, {{ 9,  1,  0}, { 3,  2, 11}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{11,  1,  2}, { 9,  1, 11}, { 8,  9, 11}, {-1, -1, -1}, {-1, -1, -1}}},
	{2, {{10,  3,  1}, {10, 11,  3}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{10,  0,  1}, { 8,  0, 10}, {11,  8, 10}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 9,  3,  0}, {11,  3,  9}, {10, 11,  9}, {-1, -1, -1}, {-1, -1, -1}}},
	{2, {{ 8,  9, 10}, { 8, 10, 11}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{1, {{ 7,  4,  8}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{2, {{ 3,  4,  0}, { 3,  7,  4}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{2, {{ 1,  0,  9}, { 4,  8,  7}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 1,  4,  9}, { 7,  4,  1}, { 3,  7,  1}, {-1, -1, -1}, {-1, -1, -1}}},
	{2, {{ 2,  1, 10}, { 4,  8,  7}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 4,  3,  7}, { 0,  3,  4}, { 2,  1, 10}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 2,  9, 10}, { 0,  9,  2}, { 4,  8,  7}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{10,  2,  9}, { 9,  2,  7}, { 7,  2,  3}, { 9,  7,  4}, {-1, -1, -1}}},
	{2, {{ 4,  8,  7}, {11,  3,  2}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 4, 11,  7}, { 2, 11,  4}, { 0,  2,  4}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 0,  9,  1}, { 4,  8,  7}, { 3,  2, 11}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{ 7,  4, 11}, { 4,  9, 11}, {11,  9,  2}, { 2,  9,  1}, {-1, -1, -1}}},
	{3, {{10,  3,  1}, {11,  3, 10}, { 8,  7,  4}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{11,  1, 10}, { 4,  1, 11}, { 0,  1,  4}, {11,  7,  4}, {-1, -1, -1}}},
	{4, {{ 7,  4,  8}, { 0,  9, 11}, {11,  9, 10}, { 0, 11,  3}, {-1, -1, -1}}},
	{3, {{ 7,  4, 11}, {11,  4,  9}, {11,  9, 10}, {-1, -1, -1}, {-1, -1, -1}}},
	{1, {{ 5,  9,  4}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{2, {{ 5,  9,  4}, { 8,  0,  3}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{2, {{ 5,  0,  4}, { 5,  1,  0}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 5,  8,  4}, { 3,  8,  5}, { 1,  3,  5}, {-1, -1, -1}, {-1, -1, -1}}},
	{2, {{ 2,  1, 10}, { 5,  9,  4}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 0,  3,  8}, { 2,  1, 10}, { 9,  4,  5}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 2,  5, 10}, { 4,  5,  2}, { 0,  4,  2}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{10,  2,  5}, { 2,  3,  5}, { 5,  3,  4}, { 4,  3,  8}, {-1, -1, -1}}},
	{2, {{ 5,  9,  4}, { 3,  2, 11}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{11,  0,  2}, { 8,  0, 11}, { 9,  4,  5}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 5,  0,  4}, { 1,  0,  5}, { 3,  2, 11}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{ 1,  2,  5}, { 5,  2,  8}, { 8,  2, 11}, { 8,  4,  5}, {-1, -1, -1}}},
	{3, {{ 3, 10, 11}, { 1, 10,  3}, { 5,  9,  4}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{ 9,  4,  5}, { 8,  0,  1}, {10,  8,  1}, {11,  8, 10}, {-1, -1, -1}}},
	{4, {{ 4,  5,  0}, { 0,  5, 11}, {11,  5, 10}, { 0, 11,  3}, {-1, -1, -1}}},
	{3, {{ 4,  5,  8}, { 8,  5, 10}, { 8, 10, 11}, {-1, -1, -1}, {-1, -1, -1}}},
	{2, {{ 7,  9,  8}, { 7,  5,  9}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 3,  9,  0}, { 5,  9,  3}, { 7,  5,  3}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 7,  0,  8}, { 1,  0,  7}, { 5,  1,  7}, {-1, -1, -1}, {-1, -1, -1}}},
	{2, {{ 5,  1,  3}, { 5,  3,  7}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 7,  9,  8}, { 5,  9,  7}, { 1, 10,  2}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{ 1, 10,  2}, { 5,  9,  0}, { 3,  5,  0}, { 7,  5,  3}, {-1, -1, -1}}},
	{4, {{ 0,  8,  2}, { 2,  8,  5}, { 5,  8,  7}, { 5, 10,  2}, {-1, -1, -1}}},
	{3, {{10,  2,  5}, { 5,  2,  3}, { 5,  3,  7}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 9,  7,  5}, { 8,  7,  9}, {11,  3,  2}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{ 5,  9,  7}, { 7,  9,  2}, { 2,  9,  0}, { 7,  2, 11}, {-1, -1, -1}}},
	{4, {{ 3,  2, 11}, { 1,  0,  8}, { 7,  1,  8}, { 5,  1,  7}, {-1, -1, -1}}},
	{3, {{ 2, 11,  1}, { 1, 11,  7}, { 1,  7,  5}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{ 5,  9,  8}, { 5,  8,  7}, { 1, 10,  3}, { 3, 10, 11}, {-1, -1, -1}}},
	{5, {{ 7,  5,  0}, { 0,  5,  9}, {11,  7,  0}, { 0,  1, 10}, {10, 11,  0}}},
	{5, {{10, 11,  0}, { 0, 11,  3}, { 5, 10,  0}, { 0,  8,  7}, { 7,  5,  0}}},
	{2, {{10, 11,  5}, {11,  7,  5}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{1, {{ 6, 10,  5}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{2, {{ 8,  0,  3}, {10,  5,  6}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{2, {{ 0,  9,  1}, {10,  5,  6}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 8,  1,  3}, { 9,  1,  8}, {10,  5,  6}, {-1, -1, -1}, {-1, -1, -1}}},
	{2, {{ 6,  1,  5}, { 6,  2,  1}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 6,  1,  5}, { 2,  1,  6}, { 0,  3,  8}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 6,  9,  5}, { 0,  9,  6}, { 2,  0,  6}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{ 9,  5,  8}, { 8,  5,  2}, { 2,  5,  6}, { 2,  3,  8}, {-1, -1, -1}}},
	{2, {{ 3,  2, 11}, { 6, 10,  5}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 0, 11,  8}, { 2, 11,  0}, { 6, 10,  5}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 1,  0,  9}, { 3,  2, 11}, {10,  5,  6}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{10,  5,  6}, { 9,  1,  2}, {11,  9,  2}, { 8,  9, 11}, {-1, -1, -1}}},
	{3, {{ 3,  6, 11}, { 5,  6,  3}, { 1,  5,  3}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{ 8,  0, 11}, {11,  0,  5}, { 5,  0,  1}, {11,  5,  6}, {-1, -1, -1}}},
	{4, {{11,  3,  6}, { 3,  0,  6}, { 6,  0,  5}, { 5,  0,  9}, {-1, -1, -1}}},
	{3, {{ 5,  6,  9}, { 9,  6, 11}, { 9, 11,  8}, {-1, -1, -1}, {-1, -1, -1}}},
	{2, {{10,  5,  6}, { 7,  4,  8}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 3,  4,  0}, { 7,  4,  3}, { 5,  6, 10}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 9,  1,  0}, {10,  5,  6}, { 4,  8,  7}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{ 6, 10,  5}, { 9,  1,  7}, { 7,  1,  3}, { 9,  7,  4}, {-1, -1, -1}}},
	{3, {{ 1,  6,  2}, { 5,  6,  1}, { 7,  4,  8}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{ 2,  1,  5}, { 2,  5,  6}, { 0,  3,  4}, { 4,  3,  7}, {-1, -1, -1}}},
	{4, {{ 4,  8,  7}, { 0,  9,  5}, { 6,  0,  5}, { 2,  0,  6}, {-1, -1, -1}}},
	{5, {{ 3,  7,  9}, { 9,  7,  4}, { 2,  3,  9}, { 9,  5,  6}, { 6,  2,  9}}},
	{3, {{11,  3,  2}, { 8,  7,  4}, { 6, 10,  5}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{10,  5,  6}, { 7,  4,  2}, { 2,  4,  0}, { 7,  2, 11}, {-1, -1, -1}}},
	{4, {{ 1,  0,  9}, { 7,  4,  8}, { 3,  2, 11}, {10,  5,  6}, {-1, -1, -1}}},
	{5, {{ 2,  9,  1}, {11,  9,  2}, { 4,  9, 11}, {11,  7,  4}, {10,  5,  6}}},
	{4, {{ 4,  8,  7}, {11,  3,  5}, { 5,  3,  1}, {11,  5,  6}, {-1, -1, -1}}},
	{5, {{ 1,  5, 11}, {11,  5,  6}, { 0,  1, 11}, {11,  7,  4}, { 4,  0, 11}}},
	{5, {{ 5,  0,  9}, { 6,  0,  5}, { 3,  0,  6}, { 6, 11,  3}, { 4,  8,  7}}},
	{4, {{ 5,  6,  9}, { 9,  6, 11}, { 7,  4,  9}, {11,  7,  9}, {-1, -1, -1}}},
	{2, {{ 4, 10,  9}, { 4,  6, 10}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{10,  4,  6}, { 9,  4, 10}, { 8,  0,  3}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 0, 10,  1}, { 6, 10,  0}, { 4,  6,  0}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{ 3,  8,  1}, { 1,  8,  6}, { 6,  8,  4}, { 1,  6, 10}, {-1, -1, -1}}},
	{3, {{ 4,  1,  9}, { 2,  1,  4}, { 6,  2,  4}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{ 0,  3,  8}, { 2,  1,  9}, { 4,  2,  9}, { 6,  2,  4}, {-1, -1, -1}}},
	{2, {{ 2,  0,  4}, { 2,  4,  6}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 3,  8,  2}, { 2,  8,  4}, { 2,  4,  6}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 4, 10,  9}, { 6, 10,  4}, { 2, 11,  3}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{ 8,  0,  2}, { 8,  2, 11}, { 9,  4, 10}, {10,  4,  6}, {-1, -1, -1}}},
	{4, {{11,  3,  2}, { 1,  0,  6}, { 6,  0,  4}, { 1,  6, 10}, {-1, -1, -1}}},
	{5, {{ 4,  6,  1}, { 1,  6, 10}, { 8,  4,  1}, { 1,  2, 11}, {11,  8,  1}}},
	{4, {{ 6,  9,  4}, { 3,  9,  6}, { 1,  9,  3}, { 6, 11,  3}, {-1, -1, -1}}},
	{5, {{11,  8,  1}, { 1,  8,  0}, { 6, 11,  1}, { 1,  9,  4}, { 4,  6,  1}}},
	{3, {{11,  3,  6}, { 6,  3,  0}, { 6,  0,  4}, {-1, -1, -1}, {-1, -1, -1}}},
	{2, {{ 4,  6,  8}, { 6, 11,  8}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{10,  7,  6}, { 8,  7, 10}, { 9,  8, 10}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{ 7,  0,  3}, {10,  0,  7}, { 9,  0, 10}, { 7,  6, 10}, {-1, -1, -1}}},
	{4, {{ 6, 10,  7}, {10,  1,  7}, { 7,  1,  8}, { 8,  1,  0}, {-1, -1, -1}}},
	{3, {{ 6, 10,  7}, { 7, 10,  1}, { 7,  1,  3}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{ 2,  1,  6}, { 6,  1,  8}, { 8,  1,  9}, { 6,  8,  7}, {-1, -1, -1}}},
	{5, {{ 6,  2,  9}, { 9,  2,  1}, { 7,  6,  9}, { 9,  0,  3}, { 3,  7,  9}}},
	{3, {{ 8,  7,  0}, { 0,  7,  6}, { 0,  6,  2}, {-1, -1, -1}, {-1, -1, -1}}},
	{2, {{ 3,  7,  2}, { 7,  6,  2}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{ 3,  2, 11}, { 6, 10,  8}, { 8, 10,  9}, { 6,  8,  7}, {-1, -1, -1}}},
	{5, {{ 0,  2,  7}, { 7,  2, 11}, { 9,  0,  7}, { 7,  6, 10}, {10,  9,  7}}},
	{5, {{ 8,  1,  0}, { 7,  1,  8}, {10,  1,  7}, { 7,  6, 10}, { 3,  2, 11}}},
	{4, {{ 2, 11,  1}, { 1, 11,  7}, { 6, 10,  1}, { 7,  6,  1}, {-1, -1, -1}}},
	{5, {{ 9,  8,  6}, { 6,  8,  7}, { 1,  9,  6}, { 6, 11,  3}, { 3,  1,  6}}},
	{2, {{ 9,  0,  1}, { 6, 11,  7}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{ 8,  7,  0}, { 0,  7,  6}, {11,  3,  0}, { 6, 11,  0}, {-1, -1, -1}}},
	{1, {{11,  7,  6}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{1, {{ 6,  7, 11}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{2, {{ 0,  3,  8}, { 7, 11,  6}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{2, {{ 1,  0,  9}, { 7, 11,  6}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 1,  8,  9}, { 3,  8,  1}, { 7, 11,  6}, {-1, -1, -1}, {-1, -1, -1}}},
	{2, {{ 1, 10,  2}, {11,  6,  7}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 2,  1, 10}, { 0,  3,  8}, {11,  6,  7}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 9,  2,  0}, {10,  2,  9}, {11,  6,  7}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{11,  6,  7}, {10,  2,  3}, { 8, 10,  3}, { 9, 10,  8}, {-1, -1, -1}}},
	{2, {{ 2,  7,  3}, { 2,  6,  7}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 0,  7,  8}, { 6,  7,  0}, { 2,  6,  0}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 7,  2,  6}, { 3,  2,  7}, { 1,  0,  9}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{ 6,  1,  2}, { 8,  1,  6}, { 9,  1,  8}, { 7,  8,  6}, {-1, -1, -1}}},
	{3, {{ 7, 10,  6}, { 1, 10,  7}, { 3,  1,  7}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{ 7, 10,  6}, { 7,  1, 10}, { 8,  1,  7}, { 0,  1,  8}, {-1, -1, -1}}},
	{4, {{ 3,  0,  7}, { 7,  0, 10}, {10,  0,  9}, {10,  6,  7}, {-1, -1, -1}}},
	{3, {{ 6,  7, 10}, {10,  7,  8}, {10,  8,  9}, {-1, -1, -1}, {-1, -1, -1}}},
	{2, {{ 8,  6,  4}, { 8, 11,  6}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 6,  3, 11}, { 0,  3,  6}, { 4,  0,  6}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 6,  8, 11}, { 4,  8,  6}, { 0,  9,  1}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{ 4,  9,  6}, { 6,  9,  3}, { 3,  9,  1}, { 3, 11,  6}, {-1, -1, -1}}},
	{3, {{ 8,  6,  4}, {11,  6,  8}, {10,  2,  1}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{ 2,  1, 10}, { 0,  3, 11}, { 6,  0, 11}, { 4,  0,  6}, {-1, -1, -1}}},
	{4, {{11,  4,  8}, { 6,  4, 11}, { 2,  0,  9}, {10,  2,  9}, {-1, -1, -1}}},
	{5, {{ 9, 10,  3}, { 3, 10,  2}, { 4,  9,  3}, { 3, 11,  6}, { 6,  4,  3}}},
	{3, {{ 2,  8,  3}, { 4,  8,  2}, { 6,  4,  2}, {-1, -1, -1}, {-1, -1, -1}}},
	{2, {{ 4,  0,  2}, { 6,  4,  2}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{ 9,  1,  0}, { 3,  2,  4}, { 4,  2,  6}, { 3,  4,  8}, {-1, -1, -1}}},
	{3, {{ 9,  1,  4}, { 4,  1,  2}, { 4,  2,  6}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{ 1,  8,  3}, { 6,  8,  1}, { 4,  8,  6}, {10,  6,  1}, {-1, -1, -1}}},
	{3, {{ 1, 10,  0}, { 0, 10,  6}, { 0,  6,  4}, {-1, -1, -1}, {-1, -1, -1}}},
	{5, {{ 6,  4,  3}, { 3,  4,  8}, {10,  6,  3}, { 3,  0,  9}, { 9, 10,  3}}},
	{2, {{ 9, 10,  4}, {10,  6,  4}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{2, {{ 9,  4,  5}, { 6,  7, 11}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 8,  0,  3}, { 9,  4,  5}, { 7, 11,  6}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 0,  5,  1}, { 4,  5,  0}, { 6,  7, 11}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{ 7, 11,  6}, { 3,  8,  4}, { 5,  3,  4}, { 1,  3,  5}, {-1, -1, -1}}},
	{3, {{ 5,  9,  4}, { 1, 10,  2}, { 6,  7, 11}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{11,  6,  7}, { 2,  1, 10}, { 8,  0,  3}, { 9,  4,  5}, {-1, -1, -1}}},
	{4, {{ 6,  7, 11}, { 4,  5, 10}, { 2,  4, 10}, { 0,  4,  2}, {-1, -1, -1}}},
	{5, {{ 4,  3,  8}, { 5,  3,  4}, { 2,  3,  5}, { 5, 10,  2}, { 7, 11,  6}}},
	{3, {{ 2,  7,  3}, { 6,  7,  2}, { 4,  5,  9}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{ 5,  9,  4}, { 8,  0,  6}, { 6,  0,  2}, { 8,  6,  7}, {-1, -1, -1}}},
	{4, {{ 6,  3,  2}, { 7,  3,  6}, { 5,  1,  0}, { 4,  5,  0}, {-1, -1, -1}}},
	{5, {{ 2,  6,  8}, { 8,  6,  7}, { 1,  2,  8}, { 8,  4,  5}, { 5,  1,  8}}},
	{4, {{ 5,  9,  4}, { 1, 10,  6}, { 7,  1,  6}, { 3,  1,  7}, {-1, -1, -1}}},
	{5, {{ 6,  1, 10}, { 7,  1,  6}, { 0,  1,  7}, { 7,  8,  0}, { 5,  9,  4}}},
	{5, {{ 0,  4, 10}, {10,  4,  5}, { 3,  0, 10}, {10,  6,  7}, { 7,  3, 10}}},
	{4, {{ 6,  7, 10}, {10,  7,  8}, { 4,  5, 10}, { 8,  4, 10}, {-1, -1, -1}}},
	{3, {{ 9,  6,  5}, {11,  6,  9}, { 8, 11,  9}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{ 6,  3, 11}, { 6,  0,  3}, { 5,  0,  6}, { 9,  0,  5}, {-1, -1, -1}}},
	{4, {{11,  0,  8}, { 5,  0, 11}, { 1,  0,  5}, { 6,  5, 11}, {-1, -1, -1}}},
	{3, {{11,  6,  3}, { 3,  6,  5}, { 3,  5,  1}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{ 2,  1, 10}, { 5,  9, 11}, {11,  9,  8}, { 5, 11,  6}, {-1, -1, -1}}},
	{5, {{11,  0,  3}, { 6,  0, 11}, { 9,  0,  6}, { 6,  5,  9}, { 2,  1, 10}}},
	{5, {{ 8, 11,  5}, { 5, 11,  6}, { 0,  8,  5}, { 5, 10,  2}, { 2,  0,  5}}},
	{4, {{11,  6,  3}, { 3,  6,  5}, {10,  2,  3}, { 5, 10,  3}, {-1, -1, -1}}},
	{4, {{ 8,  5,  9}, { 2,  5,  8}, { 6,  5,  2}, { 8,  3,  2}, {-1, -1, -1}}},
	{3, {{ 5,  9,  6}, { 6,  9,  0}, { 6,  0,  2}, {-1, -1, -1}, {-1, -1, -1}}},
	{5, {{ 5,  1,  8}, { 8,  1,  0}, { 6,  5,  8}, { 8,  3,  2}, { 2,  6,  8}}},
	{2, {{ 5,  1,  6}, { 1,  2,  6}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{5, {{ 3,  1,  6}, { 6,  1, 10}, { 8,  3,  6}, { 6,  5,  9}, { 9,  8,  6}}},
	{4, {{ 1, 10,  0}, { 0, 10,  6}, { 5,  9,  0}, { 6,  5,  0}, {-1, -1, -1}}},
	{2, {{ 3,  0,  8}, { 6,  5, 10}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{1, {{ 5, 10,  6}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{2, {{ 5, 11, 10}, { 5,  7, 11}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 5, 11, 10}, { 7, 11,  5}, { 3,  8,  0}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{11,  5,  7}, {10,  5, 11}, { 9,  1,  0}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{ 7, 10,  5}, {11, 10,  7}, { 8,  9,  1}, { 3,  8,  1}, {-1, -1, -1}}},
	{3, {{ 1, 11,  2}, { 7, 11,  1}, { 5,  7,  1}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{ 8,  0,  3}, { 2,  1,  7}, { 7,  1,  5}, { 2,  7, 11}, {-1, -1, -1}}},
	{4, {{ 7,  9,  5}, { 2,  9,  7}, { 0,  9,  2}, {11,  2,  7}, {-1, -1, -1}}},
	{5, {{ 5,  7,  2}, { 2,  7, 11}, { 9,  5,  2}, { 2,  3,  8}, { 8,  9,  2}}},
	{3, {{ 5,  2, 10}, { 3,  2,  5}, { 7,  3,  5}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{ 2,  8,  0}, { 5,  8,  2}, { 7,  8,  5}, { 2, 10,  5}, {-1, -1, -1}}},
	{4, {{ 0,  9,  1}, {10,  5,  3}, { 3,  5,  7}, {10,  3,  2}, {-1, -1, -1}}},
	{5, {{ 8,  9,  2}, { 2,  9,  1}, { 7,  8,  2}, { 2, 10,  5}, { 5,  7,  2}}},
	{2, {{ 3,  1,  5}, { 7,  3,  5}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 8,  0,  7}, { 7,  0,  1}, { 7,  1,  5}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 0,  9,  3}, { 3,  9,  5}, { 3,  5,  7}, {-1, -1, -1}, {-1, -1, -1}}},
	{2, {{ 8,  9,  7}, { 9,  5,  7}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 8,  5,  4}, {10,  5,  8}, {11, 10,  8}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{ 0,  5,  4}, {11,  5,  0}, {10,  5, 11}, { 3, 11,  0}, {-1, -1, -1}}},
	{4, {{ 1,  0,  9}, { 4,  8, 10}, {10,  8, 11}, { 4, 10,  5}, {-1, -1, -1}}},
	{5, {{11, 10,  4}, { 4, 10,  5}, { 3, 11,  4}, { 4,  9,  1}, { 1,  3,  4}}},
	{4, {{ 5,  2,  1}, { 8,  2,  5}, {11,  2,  8}, { 5,  4,  8}, {-1, -1, -1}}},
	{5, {{ 4,  0, 11}, {11,  0,  3}, { 5,  4, 11}, {11,  2,  1}, { 1,  5, 11}}},
	{5, {{ 2,  0,  5}, { 5,  0,  9}, {11,  2,  5}, { 5,  4,  8}, { 8, 11,  5}}},
	{2, {{ 4,  9,  5}, {11,  2,  3}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{ 5,  2, 10}, { 5,  3,  2}, { 4,  3,  5}, { 8,  3,  4}, {-1, -1, -1}}},
	{3, {{10,  5,  2}, { 2,  5,  4}, { 2,  4,  0}, {-1, -1, -1}, {-1, -1, -1}}},
	{5, {{10,  3,  2}, { 5,  3, 10}, { 8,  3,  5}, { 5,  4,  8}, { 1,  0,  9}}},
	{4, {{10,  5,  2}, { 2,  5,  4}, { 9,  1,  2}, { 4,  9,  2}, {-1, -1, -1}}},
	{3, {{ 4,  8,  5}, { 5,  8,  3}, { 5,  3,  1}, {-1, -1, -1}, {-1, -1, -1}}},
	{2, {{ 4,  0,  5}, { 0,  1,  5}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{ 4,  8,  5}, { 5,  8,  3}, { 0,  9,  5}, { 3,  0,  5}, {-1, -1, -1}}},
	{1, {{ 4,  9,  5}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{11,  4,  7}, { 9,  4, 11}, {10,  9, 11}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{ 8,  0,  3}, { 9,  4,  7}, {11,  9,  7}, {10,  9, 11}, {-1, -1, -1}}},
	{4, {{10,  1, 11}, {11,  1,  4}, { 4,  1,  0}, { 4,  7, 11}, {-1, -1, -1}}},
	{5, {{ 1,  3,  4}, { 4,  3,  8}, {10,  1,  4}, { 4,  7, 11}, {11, 10,  4}}},
	{4, {{11,  4,  7}, {11,  9,  4}, { 2,  9, 11}, { 1,  9,  2}, {-1, -1, -1}}},
	{5, {{ 7,  9,  4}, {11,  9,  7}, { 1,  9, 11}, {11,  2,  1}, { 8,  0,  3}}},
	{3, {{ 7, 11,  4}, { 4, 11,  2}, { 4,  2,  0}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{ 7, 11,  4}, { 4, 11,  2}, { 3,  8,  4}, { 2,  3,  4}, {-1, -1, -1}}},
	{4, {{ 9,  2, 10}, { 7,  2,  9}, { 3,  2,  7}, { 4,  7,  9}, {-1, -1, -1}}},
	{5, {{10,  9,  7}, { 7,  9,  4}, { 2, 10,  7}, { 7,  8,  0}, { 0,  2,  7}}},
	{5, {{ 7,  3, 10}, {10,  3,  2}, { 4,  7, 10}, {10,  1,  0}, { 0,  4, 10}}},
	{2, {{10,  1,  2}, { 7,  8,  4}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 9,  4,  1}, { 1,  4,  7}, { 1,  7,  3}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{ 9,  4,  1}, { 1,  4,  7}, { 8,  0,  1}, { 7,  8,  1}, {-1, -1, -1}}},
	{2, {{ 0,  4,  3}, { 4,  7,  3}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{1, {{ 8,  4,  7}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{2, {{10,  9,  8}, {11, 10,  8}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 0,  3,  9}, { 9,  3, 11}, { 9, 11, 10}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 1,  0, 10}, {10,  0,  8}, {10,  8, 11}, {-1, -1, -1}, {-1, -1, -1}}},
	{2, {{ 1,  3, 10}, { 3, 11, 10}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 2,  1, 11}, {11,  1,  9}, {11,  9,  8}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{ 0,  3,  9}, { 9,  3, 11}, { 2,  1,  9}, {11,  2,  9}, {-1, -1, -1}}},
	{2, {{ 2,  0, 11}, { 0,  8, 11}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{1, {{ 2,  3, 11}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{3, {{ 3,  2,  8}, { 8,  2, 10}, { 8, 10,  9}, {-1, -1, -1}, {-1, -1, -1}}},
	{2, {{10,  9,  2}, { 9,  0,  2}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{4, {{ 3,  2,  8}, { 8,  2, 10}, { 1,  0,  8}, {10,  1,  8}, {-1, -1, -1}}},
	{1, {{10,  1,  2}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{2, {{ 3,  1,  8}, { 1,  9,  8}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{1, {{ 9,  0,  1}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{1, {{ 3,  0,  8}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}},
	{0, {{-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}, {-1, -1, -1}}}
};




} // namespace
